// scrape-agents.js

const fs        = require('fs');
const path      = require('path');
const os        = require('os');
const mysql     = require('mysql2/promise');
const puppeteer = require('puppeteer-extra');
const Stealth   = require('puppeteer-extra-plugin-stealth');

puppeteer.use(Stealth());
const delay = ms => new Promise(res => setTimeout(res, ms));

// —— Kaynak adı dinamik veya sabit ——
const SOURCE_NAME = 'Sahibinden';

// —— MySQL bağlantı ayarları ——
const dbConfig = {
  host:     'localhost',
  user:     'root',
  password: '',
  database: 'emlak_db',
  charset:  'utf8mb4'
};

;(async () => {
  // 1) detailLinks.json’den linkleri al
  const cachePath = path.resolve(__dirname, 'detailLinks.json');
  if (!fs.existsSync(cachePath)) {
    console.error('Error: detailLinks.json bulunamadı.');
    process.exit(1);
  }
  const links = JSON.parse(fs.readFileSync(cachePath, 'utf8')).slice(0, 10);

  // 2) DB’ye bağlan
  const conn = await mysql.createConnection(dbConfig);
  console.log('✅ MySQL’e bağlanıldı.');

  // 3) Puppeteer’i başlat
  const browser = await puppeteer.launch({
    headless: false,
    userDataDir: path.join(
      os.platform()==='win32'
        ? process.env.LOCALAPPDATA+'\\Google\\Chrome\\User Data\\Default'
        : path.join(os.homedir(), '.config','google-chrome','Default')
    ),
    args: ['--no-sandbox','--disable-setuid-sandbox','--disable-blink-features=AutomationControlled']
  });
  const page = await browser.newPage();
  await page.setViewport({ width: 1200, height: 900 });

  // 4) Sahibinden ana sayfa (CF atlatma)
  await page.goto('https://www.sahibinden.com', { waitUntil: 'networkidle2' });
  await delay(3000);

  for (let detailUrl of links) {
    console.log('▶ Detay:', detailUrl);
    await page.goto(detailUrl, { waitUntil: 'networkidle2' });
    await delay(1500);

    // — Acente URL, ad, konum —
    const agentHref = await page.$eval(
      'div.contact-info a.store-logo',
      a => a.href
    ).catch(() => null);
    if (!agentHref) {
      console.log('  ⚠️ Acente URL bulunamadı, atlanıyor.');
      continue;
    }

    const storeDiv = await page.$('div.contact-info div.store-name');
    const agencyName = storeDiv
      ? await storeDiv.evaluate(el => el.textContent.trim())
      : '';
    const location = storeDiv
      ? await storeDiv.evaluate(el => el.getAttribute('data-main-location'))
      : '';

    console.log(`  → ${agencyName} (${location})`);

    //  — DB: acente kaydı ekle —
    const [agentRes] = await conn.execute(
      `INSERT INTO agents (source, agency_name, agent_url, location)
       VALUES (?, ?, ?, ?)`,
      [SOURCE_NAME, agencyName, agentHref, location]
    );
    const agentId = agentRes.insertId;

    // 5) Ekibimiz sayfası
    const { origin } = new URL(agentHref);
    await page.goto(`${origin}/ekibimiz`, { waitUntil: 'networkidle2' });
    await delay(3000);

    // 6) Ekip üyesi verilerini çek
    const members = await page.$$eval(
      'div.edr-trigger.consultant',
      cards => cards.map(c => ({
        name:  c.querySelector('p.name')?.innerText.trim()  || '',
        tag:   c.querySelector('p.title')?.innerText.trim() || '',
        phone: c.querySelector('a.phone-trigger')?.getAttribute('data-phone') || ''
      }))
    );

    //  — DB: ekip üyelerini ekle —
    for (let m of members) {
      await conn.execute(
        `INSERT INTO team_members (agent_id, name, tag, phone)
         VALUES (?, ?, ?, ?)`,
        [agentId, m.name, m.tag, m.phone]
      );
    }
    console.log(`  ✅ ${members.length} ekip üyesi kaydedildi.`);
  }

  await browser.close();
  await conn.end();
  console.log('🎉 İşlem tamamlandı. Veriler emlak_db içinde saklandı.');
})();
